/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xpattern.c
 * pattern term related functions
 * -----------------------------------------------------------------*/

#include <xpattern.h>
#include <xerrcapi.h>
#include <xsetting.h>

#include <xpatlex.h>


/* Pointer to database */
#define DB (bx_handlearray[handle].db)

/* Value of pattern string */
#define PATTERN(prop) (DB->Pattern[prop])

static bx_errtype BestXPattSetProg(
  bx_handletype handle,
  bx_patttype prop,
  bx_charptrtype pattern,
  bx_int32 write
);


/*---------------------------------------------------------------------------*
 * BestXPattProg
 *
 * Purpose: Sets a pattern term in host DB and writes it to card
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPattProg(
  bx_handletype handle,
  bx_patttype prop,
  bx_charptrtype pattern)

{
  BX_DECLARE_FUNCNAME("BestXPattProg [pattprog]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  { 
    BX_TRY(BestXPattSetProg(handle,prop,pattern,1 /* write to card */));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXPattSet
 *
 * Purpose: Sets a pattern term in host DB and checks syntax without
 *          writing to card
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPattSet(
  bx_handletype handle,
  bx_patttype prop,
  bx_charptrtype pattern)

{
  BX_DECLARE_FUNCNAME("BestXPattSet [pattset]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  { 
    BX_TRY(BestXPattSetProg(handle,prop,pattern,0 /* do not write to card */));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXPattSetProg
 *
 * Purpose: Sets a pattern term in host DB and optionally writes it to card
 *---------------------------------------------------------------------------*/

static bx_errtype BestXPattSetProg(
  bx_handletype handle,
  bx_patttype prop,
  bx_charptrtype pattern,
  bx_int32 write)

{
  BX_DECLARE_FUNCNAME("BestXPattProg [pattprog]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  { 
    /* Range checking */
    BX_FCT_PARAM_CHK(1,prop>=BX_PATT_ENUMSIZE);
    if (BestXHasMephisto(handle) && write)
    {
      /* Mephisto has not BUS2,BUS3,OBS4,OBS5 pattern terms */
      BX_FCT_PARAM_CHK(1,prop==BX_PATT_BUS2);
      BX_FCT_PARAM_CHK(1,prop==BX_PATT_BUS3);
      BX_FCT_PARAM_CHK(1,prop==BX_PATT_OBS4);
      BX_FCT_PARAM_CHK(1,prop==BX_PATT_OBS5);
    }

    BX_TRY_FCT_PARAM_NULL_POINTER(pattern);
    
    /* Parse and optionally write to card */
    BX_TRY(parse_eqn (handle, pattern, prop, write));

    if (PATTERN(prop))
    {
      /* first free the old string */
      BestXMemFree((void **) &(PATTERN(prop))); 
      PATTERN(prop)=NULL;
    }
    
    /* Allocate the new string */
    PATTERN(prop)=(bx_charptrtype) BestXMemCalloc(BESTX_STRLEN(pattern)+1,sizeof(bx_int8));
    BX_TRY_FAIL((PATTERN(prop))?BX_E_OK:BX_E_HOST_MEM_FULL);
    
    /* Set property value */
    BESTX_STRCPY(PATTERN(prop),pattern);
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXPattGet
 *
 * Purpose: Gets a pattern term from host DB
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPattGet(
  bx_handletype handle,
  bx_patttype prop,
  bx_charptrtype *pattern)

{ 
  BX_DECLARE_FUNCNAME("BestXPattGet [pattget]");
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* Range checking */
    BX_FCT_PARAM_CHK(1,prop>=BX_PATT_ENUMSIZE);
    BX_TRY_FCT_PARAM_NULL_POINTER(pattern);
     
    /* get property */
    *pattern=(bx_charptrtype)PATTERN(prop);
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXPattDefaultSet
 *
 * Purpose: Sets all pattern terms in host DB to default
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPattDefaultSet(
  bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXPattDefaultSet [pattdefset]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  { 
    /* Trigger on !FRAME (BUS0) store all (OBS1) */
    BX_TRY(BestXPattSet(handle,BX_PATT_BUS0,"FRAME=0")); /* Trigger */
    BX_TRY(BestXPattSet(handle,BX_PATT_BUS1,"0"));
    BX_TRY(BestXPattSet(handle,BX_PATT_BUS2,"0"));
    BX_TRY(BestXPattSet(handle,BX_PATT_BUS3,"0"));
    BX_TRY(BestXPattSet(handle,BX_PATT_OBS0,"1"));       /* Storage */
    BX_TRY(BestXPattSet(handle,BX_PATT_OBS1,"0"));
    BX_TRY(BestXPattSet(handle,BX_PATT_OBS2,"0"));
    BX_TRY(BestXPattSet(handle,BX_PATT_OBS3,"1"));       /* Trigger */
    BX_TRY(BestXPattSet(handle,BX_PATT_OBS4,"0"));
    BX_TRY(BestXPattSet(handle,BX_PATT_OBS5,"0"));
    BX_TRY(BestXPattSet(handle,BX_PATT_ERR0,"0"));
    BX_TRY(BestXPattSet(handle,BX_PATT_TRAN0,"T(FRAME)"));
    BX_TRY(BestXPattSet(handle,BX_PATT_COND0,"1"));
    BX_TRY(BestXPattSet(handle,BX_PATT_COND1,"1"));
  }

  BX_ERRETURN(BX_TRY_RET);
}



bx_errtype BestXPattGetEdit(bx_handletype handle, bx_int32 patt_id, bx_ccharptrtype * pattern)
{
  BX_DECLARE_FUNCNAME("BestXPattGetEdit");

  bx_int8 pid = (bx_int8) patt_id;
#if 0 /* chris TBD */
  
  bx_int16 out_size = OUT_PATT_GET;
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_ANALYZER|BX_CAPABILITY_CAPI);
    BX_TRY_FCT_PARAM_NULL_POINTER(pattern);
    BX_TRY_FCT_PARAM(patt_id, patt_id > BX_PATT_ENUMSIZE);
    BX_TRY(BestXBasicCommand(handle, CMD_PATT_EDIT_GET,
        &pid, IN_PATT_GET,
        (bx_int8ptr) tmp_ladata_buf, &out_size));
    *pattern = tmp_ladata_buf;
  }
#endif
  BX_ERRETURN(BX_E_TBD);
}
